
#
# $Id: fw_functions.pl 8 2006-02-15 12:56:25Z gregor $
#
#Copyright (c) 2006 Gregor Maier <gregor@majordomus.org>
#All rights reserved.
#
#
#Redistribution and use in source and binary forms, with or without
#modification, are permitted provided that the following conditions
#are met:
#
#1. Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#2. Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in the
#   documentation and/or other materials provided with the distribution.
#3. Neither the names of the copyright owners nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
#WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
#MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
#IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
#DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
#DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
#GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
#INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
#IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
#OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
#ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#
#

#
# Global symbols that must be defined, when calling these
# functions: 
# $LINENUM .... Linenumber in the current file
# $SUBNET ..... The current subnet
#

use strict;

my $srcport_range = "0:65535";
my $pubports = "1024:65535";

sub xxx_allow_me {
	# Traffic to/from birdcage
	allow_forward_tcpudp("tcp", "birdcage", $srcport_range, "birdcage", 22, "local_in");

	allow_forward_tcpudp("udp", "birdcage", $srcport_range, "birdcage", 53, "local_out");
	allow_forward_tcpudp("tcp", "birdcage", $srcport_range, "birdcage", 53, "local_out");
	allow_forward_tcpudp("tcp", "birdcage", $srcport_range, "birdcage", 22, "local_out");

	# for apt 
	allow_forward_tcpudp("tcp", "extern", $srcport_range, "extern", 80, "local_out");
	int_allow_forward_icmp_related("any", "any", "local_out");
	int_allow_forward_icmp_related("any", "any", "local_in");

	allow_me_icmp6();

	# Silence Virtual Router Redundancy Protocol on external link 
	# from RBG. We don't want those in out logs
	gmsystem("$IP6TABLES -A INPUT -p 112 -j DROP");

	# Silence IPP broadcasts
	gmsystem("$IPTABLES -A INPUT -p tcp --sport 631 --dport 631 -j DROP");
	gmsystem("$IP6TABLES -A INPUT -p tcp --sport 631 --dport 631 -j DROP");
	gmsystem("$IPTABLES -A INPUT -p udp --sport 631 --dport 631 -j DROP");
	gmsystem("$IP6TABLES -A INPUT -p udp --sport 631 --dport 631 -j DROP");

	gmsystem("$IPTABLES -A INPUT -j LOG --log-prefix \"DROP INPUT \"");
	gmsystem("$IP6TABLES -A INPUT -j LOG --log-prefix \"DROP INPUT \"");
	gmsystem("$IPTABLES -A OUTPUT -j LOG --log-prefix \"DROP OUTPUT \"");
	gmsystem("$IP6TABLES -A OUTPUT -j LOG --log-prefix \"DROP OUTPUT \"");

}


############################################################################
# FUNCTIONS FOR FORWARDING RULES                                           #
############################################################################

#
# Outbound for tcp traffic
#
sub
allow_out_tcp  { my ($dst, $service, $constraint) = @_;
	if ($constraint ne "") {
		allow_forward_tcpudp("tcp", $constraint, $srcport_range, $dst, $service, "out");
	} else {
		allow_forward_tcpudp("tcp", $SUBNET, $srcport_range, $dst, $service, "out");
	}
}

#
# Inbound for tcp traffic
#
sub
allow_in_tcp  { my ($src, $service, $constraint) = @_;
	if ($constraint ne "") {
		allow_forward_tcpudp("tcp", $src, $srcport_range, $constraint, $service, "in");
	} else {
		allow_forward_tcpudp("tcp", $src, $srcport_range, $SUBNET, $service, "in");
	}
}


#
# Outbound for udp traffic
#
sub
allow_out_udp { my ($dst, $service, $constraint) = @_;
	if ($constraint ne "") {
		allow_forward_tcpudp("udp", $constraint, $srcport_range, $dst, $service, "out");
	} else {
		allow_forward_tcpudp("udp", $SUBNET, $srcport_range, $dst, $service, "out");
	}
}

#
# Inbound for udp traffic
#
sub
allow_in_udp  { my ($src, $service, $constraint) = @_;
	if ($constraint ne "") {
		allow_forward_tcpudp("udp", $src, $srcport_range, $constraint, $service, "in");
	} else {
		allow_forward_tcpudp("udp", $src, $srcport_range, $SUBNET, $service, "in");
	}
}



#
# Outbound for ftp (active and passive)
#
sub
allow_out_ftp { my ($dst, $constraint) = @_;
	if ($constraint ne "") {
		allow_forward_tcpudp("tcp", $constraint, $srcport_range, $dst, 21, "out");
		int_allow_forward_ftp_passive_data($constraint, $dst, "out");
		int_allow_forward_ftp_active_data($constraint, $dst, "out");
	} else {
		allow_forward_tcpudp("tcp", $SUBNET, $srcport_range, $dst, 21, "out");
		int_allow_forward_ftp_passive_data($SUBNET, $dst, "out");
		int_allow_forward_ftp_active_data($SUBNET, $dst, "out");
	}
}
	
#
# Inbound for ftp (active and passive)
#
sub
allow_in_ftp { my ($src, $constraint) = @_;
	if ($constraint ne "") {
		allow_forward_tcpudp("tcp", $src, $srcport_range, $constraint, 21, "in");
		int_allow_forward_ftp_passive_data($src, $constraint, "in"); 
		int_allow_forward_ftp_active_data($src, $constraint, "in"); 
	} else {
		allow_forward_tcpudp("tcp", $src, $srcport_range, $SUBNET, 21, "in");
		int_allow_forward_ftp_passive_data($src, $SUBNET, "in"); 
		int_allow_forward_ftp_active_data($src, $SUBNET, "in"); 
	}
}

#
# Outbound for cisco ipsec / vpn
#
sub
allow_out_ciscovpn { my ($dst, $constraint) = @_;
	if ($constraint ne "") {
		int_allow_forward_ciscovpn($constraint, $dst, "out");
	} else {
		int_allow_forward_ciscovpn($SUBNET, $dst, "out");
	}
}

#
# Inbound for cisco ipsec / vpn
#
sub
allow_in_ciscovpn { my ($src, $constraint) = @_;
	if ($constraint ne "") {
		int_allow_forward_ciscovpn($src, $constraint, "in");
	} else {
		int_allow_forward_ciscovpn($src, $SUBNET, "in");
	}
}


#
# Outbound for related icmp traffic
# 
sub
allow_out_icmp_related { my ($dst, $constraint) = @_;
	if ($constraint ne "") {
		int_allow_forward_icmp_related($constraint, $dst, "out");
	} else { 
		int_allow_forward_icmp_related($SUBNET, $dst, "out");
	}
}

#
# Inbound for related icmp traffic
#
sub
allow_in_icmp_related { my ($src, $constraint) = @_;
	if ($constraint ne "") {
		int_allow_forward_icmp_related($src, $constraint, "in");
	} else {
		int_allow_forward_icmp_related($src, $SUBNET, "in");
	}
}


#
# Outbound ping 
#
sub
allow_out_ping { my ($dst, $constraint) = @_;
	if ($constraint ne "") {
		int_allow_forward_ping($constraint, $dst, "out");
	} else {
		int_allow_forward_ping($SUBNET, $dst, "out");
	}
}

#
# Inbound ping 
#
sub
allow_in_ping { my ($src, $constraint) = @_;
	if ($constraint ne "") {
		int_allow_forward_ping($src, $constraint, "in");
	} else {
		int_allow_forward_ping($src, $SUBNET, "in");
	}
}


#
# Outbound all
#
sub
allow_out_all { my ($dst, $constraint) = @_;
	if ($constraint ne "") {
		int_allow_forward_all($constraint, $dst, "out");
	} else {
		int_allow_forward_all($SUBNET, $dst, "out");
	}
}

#
# Inbound all 
#
sub
allow_in_all { my ($src, $constraint) = @_;
	if ($constraint ne "") {
		int_allow_forward_all($src, $constraint, "in");
	} else {
		int_allow_forward_all($src, $SUBNET, "in");
	}
}


#
# ICMPv6 to/from the local machine
# Enable Neighbor solicitation messages
# Enable outgoing routeradvertisement and incoming router solicitation
#
sub
allow_me_icmp6 {
	gmsystem("$IP6TABLES -A INPUT -p icmpv6 --icmpv6-type router-solicitation -j ACCEPT");
	gmsystem("$IP6TABLES -A OUTPUT -p icmpv6 --icmpv6-type router-advertisement -j ACCEPT");
	# Prevent received RAs to show up in the log file
	# needed because we receive our own RAs 
	gmsystem("$IP6TABLES -A INPUT -p icmpv6 --icmpv6-type router-advertisement -j DROP");

	# Multicast Listener Report
	gmsystem("$IP6TABLES -A INPUT -p icmpv6 --icmpv6-type 143 -j ACCEPT");
	gmsystem("$IP6TABLES -A OUTPUT -p icmpv6 --icmpv6-type 143 -j ACCEPT");

	gmsystem("$IP6TABLES -A INPUT -p icmpv6 --icmpv6-type neighbour-solicitation -j ACCEPT");
	gmsystem("$IP6TABLES -A OUTPUT -p icmpv6 --icmpv6-type neighbour-solicitation -j ACCEPT");

	gmsystem("$IP6TABLES -A INPUT -p icmpv6 --icmpv6-type neighbour-advertisement -j ACCEPT");
	gmsystem("$IP6TABLES -A OUTPUT -p icmpv6 --icmpv6-type neighbour-advertisement -j ACCEPT");

	# inverse neighbor solicitation
	gmsystem("$IP6TABLES -A INPUT -p icmpv6 --icmpv6-type 141 -j ACCEPT");
	gmsystem("$IP6TABLES -A OUTPUT -p icmpv6 --icmpv6-type 141 -j ACCEPT");

	# inverse neighbor advertisement
	gmsystem("$IP6TABLES -A INPUT -p icmpv6 --icmpv6-type 142 -j ACCEPT");
	gmsystem("$IP6TABLES -A OUTPUT -p icmpv6 --icmpv6-type 142 -j ACCEPT");
}

sub
allow_forward_tcpudp {
	my ($proto, $src, $sport, $dst, $dport, $direction) = @_;
	my $syn_clause;
	my $cmd;
	my ($srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6) = int_get_addresses($src,$dst);
	my ($start_chain, $start_target, $reply_chain, $reply_target) = int_get_chains($direction);


	#printf "DEBUG: $srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6\n";
	if ($proto eq "tcp") {
		$syn_clause = " ! --syn ";
	}
	elsif ($proto eq "udp") {
		$syn_clause = ""; 
	}
	else {
		print STDERR
			"ERROR: $SUBNET:$LINENUM: Protocol must be udp or tcp\n";
		return;
	}

	if ($srcaddr4 ne "" and $dstaddr4 ne "") {
		$cmd = "$IPTABLES -A $start_chain -s $srcaddr4 -d $dstaddr4 "
			. " -p $proto "
			. " --sport $sport --dport $dport "
			. " -m state --state NEW,ESTABLISHED "
			. " -j $start_target";
		gmsystem($cmd);
		$cmd = "$IPTABLES -A $reply_chain -s $dstaddr4 -d $srcaddr4 "
			. " -p $proto "
			. " --sport $dport --dport $sport "
			. " -m state --state ESTABLISHED "
			. " -j $reply_target";
		gmsystem($cmd);
	}

	if ($srcaddr6 ne "" and $dstaddr6 ne "") {
		$cmd = "$IP6TABLES -A $start_chain -s $srcaddr6 -d $dstaddr6 "
			. " -p $proto "
			. " --sport $sport --dport $dport "
			. " -m state --state NEW,ESTABLISHED "
			. " -j $start_target";
		gmsystem($cmd);
		$cmd = "$IP6TABLES -A $reply_chain -s $dstaddr6 -d $srcaddr6 "
			. " -p $proto "
			. " --sport $dport --dport $sport "
			. " -m state --state ESTABLISHED "
			#. $syn_clause
			. " -j $reply_target ";
		gmsystem($cmd);
	}
}

# allow passive ftp data connection
sub
int_allow_forward_ftp_passive_data {
	my ($src, $dst, $direction) = @_;
	my $cmd;
	my ($srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6) = int_get_addresses($src,$dst);
	my ($start_chain, $start_target, $reply_chain, $reply_target) = int_get_chains($direction);

	if ($srcaddr4 ne "" and $dstaddr4 ne "") {
		$cmd = "$IPTABLES -A $start_chain -s $srcaddr4 -d $dstaddr4 "
			. " -p tcp "
			. " --sport $pubports --dport $pubports "
			. " -m state --state RELATED,ESTABLISHED "
			. " -j $start_target";
		gmsystem($cmd);
		$cmd = "$IPTABLES -A $reply_chain -s $dstaddr4 -d $srcaddr4 "
			. " -p tcp "
			. " --sport $pubports --dport $pubports "
			. " -m state --state ESTABLISHED "
			. " -j $reply_target";
		gmsystem($cmd);
	}

	if ($srcaddr6 ne "" and $dstaddr6 ne "") {
		$cmd = "$IP6TABLES -A $start_chain -s $srcaddr6 -d $dstaddr6 "
			. " -p tcp "
			. " --sport $pubports --dport $pubports "
			. " -m state --state RELATED,ESTABLISHED "
			. " -j $start_target";
		gmsystem($cmd);
		$cmd = "$IP6TABLES -A $reply_chain -s $dstaddr6 -d $srcaddr6 "
			. " -p tcp "
			. " --sport $pubports --dport $pubports "
			. " -m state --state ESTABLISHED "
			#. " ! --syn "
			. " -j $reply_target ";
		gmsystem($cmd);
	}
}

# allow active ftp data connection
# src and dst are in the logic ot the ftp command connection. 
# i.e. the data connection is initiated by dst!
#
# IPv4 only, since we don't want to allow stateless active ftp 
sub
int_allow_forward_ftp_active_data {
	my ($src, $dst, $direction) = @_;
	my $cmd;
	my ($srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6) = int_get_addresses($src,$dst);
	my ($start_chain, $start_target, $reply_chain, $reply_target) = int_get_chains($direction);
	my $port = 20;

	if ($srcaddr4 ne "" and $dstaddr4 ne "") {
		$cmd = "$IPTABLES -A $start_chain -s $srcaddr4 -d $dstaddr4 "
			. " -p tcp "
			. " --sport $srcport_range --dport $port "
			. " -m state --state ESTABLISHED "
			. " -j $start_target";
		gmsystem($cmd);
		$cmd = "$IPTABLES -A $reply_chain -s $dstaddr4 -d $srcaddr4 "
			. " -p tcp "
			. " --sport $port --dport $srcport_range "
			. " -m state --state RELATED,ESTABLISHED "
			. " -j $reply_target";
		gmsystem($cmd);
	}
}

#timestamp-request
#timestamp-reply
#address-mask-request
#address-mask-reply
#source-quench
#redirect
#
sub
int_allow_forward_icmp_related {
	my ($src, $dst, $direction) = @_;
	my $cmd;
	my ($srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6) = int_get_addresses($src,$dst);
	my ($start_chain, $start_target, $reply_chain, $reply_target) = int_get_chains($direction);


	if ($srcaddr4 ne "" and $dstaddr4 ne "") {
		$cmd = "$IPTABLES -A $start_chain -s $srcaddr4 -d $dstaddr4 "
			. " -p icmp "
			. " --icmp-type ! redirect "
			. " -m state --state RELATED "
			. " -j $start_target";
		gmsystem($cmd);
	}
	if ($srcaddr6 ne "" and $dstaddr6 ne "") {
		$cmd = "$IP6TABLES -A $start_chain -s $srcaddr6 -d $dstaddr6 "
			. " -p icmpv6 "
			. " --icmpv6-type ! redirect "
			. " -m state --state RELATED "
			. " -j $start_target";
		gmsystem($cmd);
	}
}

sub
int_allow_forward_ping {
	my ($src, $dst, $direction) = @_;
	my $cmd;
	my ($srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6) = int_get_addresses($src,$dst);
	my ($start_chain, $start_target, $reply_chain, $reply_target) = int_get_chains($direction);


	if ($srcaddr4 ne "" and $dstaddr4 ne "") {
		$cmd = "$IPTABLES -A $start_chain -s $srcaddr4 -d $dstaddr4 "
			. " -p icmp "
			. " --icmp-type echo-request "
			. " -m state --state NEW "
			. " -j $start_target";
		gmsystem($cmd);
		$cmd = "$IPTABLES -A $reply_chain -s $dstaddr4 -d $srcaddr4 "
			. " -p icmp "
			. " --icmp-type echo-reply "
			. " -m state --state ESTABLISHED "
			. " -j $reply_target";
		gmsystem($cmd);
	}

	if ($srcaddr6 ne "" and $dstaddr6 ne "") {
		$cmd = "$IP6TABLES -A $start_chain -s $srcaddr6 -d $dstaddr6 "
			. " -p icmpv6 "
			. " --icmpv6-type echo-request "
			. " -m state --state NEW "
			. " -j $start_target";
		gmsystem($cmd);
		$cmd = "$IP6TABLES -A $reply_chain -s $dstaddr6 -d $srcaddr6 "
			. " -p icmpv6 "
			. " --icmpv6-type echo-reply "
			. " -m state --state ESTABLISHED "
			. " -j $reply_target ";
		gmsystem($cmd);
	}
}

sub
int_allow_forward_ciscovpn {
	my ($src, $dst, $direction) = @_;
	my ($cmd, $proto);
	my ($srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6) = int_get_addresses($src,$dst);
	my ($start_chain, $start_target, $reply_chain, $reply_target) = int_get_chains($direction);

	#printf "DEBUG: $srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6\n";

	# When tunneling over tcp/udp packets
	allow_forward_tcpudp("tcp", $src, $srcport_range, $dst, 10000, $direction);
	allow_forward_tcpudp("udp", $src, $srcport_range, $dst, 10000, $direction);
	# Key exchange protocol
	allow_forward_tcpudp("udp", $src, 500, $dst, 500, $direction);

	# Rules for AH and ESP protocols
	foreach $proto (50, 51) {
		if ($srcaddr4 ne "" and $dstaddr4 ne "") {
			$cmd = "$IPTABLES -A $start_chain -s $srcaddr4 -d $dstaddr4 "
				. " -p $proto "
				. " -j $start_target";
			gmsystem($cmd);
			$cmd = "$IPTABLES -A $reply_chain -s $dstaddr4 -d $srcaddr4 "
				. " -p $proto "
				. " -j $reply_target";
			gmsystem($cmd);
		}

		if ($srcaddr6 ne "" and $dstaddr6 ne "") {
			$cmd = "$IP6TABLES -A $start_chain -s $srcaddr6 -d $dstaddr6 "
				. " -p $proto "
				. " -j $start_target";
			gmsystem($cmd);
			$cmd = "$IP6TABLES -A $reply_chain -s $dstaddr6 -d $srcaddr6 "
				. " -p $proto "
				. " -j $reply_target ";
			gmsystem($cmd);
		}
	}
}

sub
int_allow_forward_all { my ($src, $dst, $direction) = @_;
	my ($srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6) = int_get_addresses($src,$dst);
	my ($start_chain, $start_target, $reply_chain, $reply_target) = int_get_chains($direction);
	my $cmd;

	if ($srcaddr4 ne "" and $dstaddr4 ne "") {
		$cmd = "$IPTABLES -A $start_chain -s $srcaddr4 -d $dstaddr4 "
			. " -j $start_target";
		gmsystem($cmd);
		$cmd = "$IPTABLES -A $reply_chain -s $dstaddr4 -d $srcaddr4 "
			. " -j $reply_target";
		gmsystem($cmd);
	}

	if ($srcaddr6 ne "" and $dstaddr6 ne "") {
		$cmd = "$IP6TABLES -A $start_chain -s $srcaddr6 -d $dstaddr6 "
			. " -j $start_target";
		gmsystem($cmd);
		$cmd = "$IP6TABLES -A $reply_chain -s $dstaddr6 -d $srcaddr6 "
			. " -j $reply_target ";
		gmsystem($cmd);
	}
}


# Convertes the src and dst specification (either the name of key from the
# CONFIG hash or a key from the IP hash) to a 4-tupel of IPv4 and IPv6 src and 
# dst addresses
sub
int_get_addresses {
	my ($src, $dst) = @_;
	my ($srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6);

	
	#print "DEBUG: $src --- $dst\n";
	if ($src =~ s/^ip:(.*)/$1/ ) { 
		$srcaddr4 =	$IP{$src}->{'net4'};
		$srcaddr6 =	$IP{$src}->{'net6'};
	} elsif ($src =~ /^any/i) {
		$srcaddr4 = "0.0.0.0/0";
		$srcaddr6 = "::/0";
	} else {
		$srcaddr4 =	$CONFIG{$src}->{'net4'};
		$srcaddr6 =	$CONFIG{$src}->{'net6'};
	}

	if ($dst =~ s/^ip:(.*)/$1/ ) { 
		$dstaddr4 =	$IP{$dst}->{'net4'};
		$dstaddr6 =	$IP{$dst}->{'net6'};
	} elsif ($dst =~ /^any/i) {
		$dstaddr4 = "0.0.0.0/0";
		$dstaddr6 = "::/0";
	} else {
		$dstaddr4 =	$CONFIG{$dst}->{'net4'};
		$dstaddr6 =	$CONFIG{$dst}->{'net6'};
	}

	# either dstaddr4/srcaddr4  or dstaddr6/srcaddr6 must be a valid tuple
	if (not 
	   (($dstaddr4 ne "" and $srcaddr4 ne "") or 
	   ($dstaddr6 ne "" and $srcaddr6 ne ""))
		){ 
		print STDERR
			"ERROR: $SUBNET:$LINENUM: No valid address has been supplied. Skipping this rule.\n";
		return;
	}
	return ($srcaddr4, $srcaddr6, $dstaddr4, $dstaddr6);
}

# Paramter: direction (in, out, local_in, local_out)
# Returns the chains needed for stateful rules. I.e. 
# start_chain: chain in which NEW connections are allowed (--state NEW,ESTABLISHED) 
# start_target: target of the rules in the start_chain if packet is allowe (XBAR or ACCEPT)
# reply_chain: chain where only ESTABLISHED packets are allowed
# reply_target: target of the reply_chain (XBAR or ACCEPT)
sub
int_get_chains {
	my $direction = shift;
	my ($start_chain, $start_target, $reply_chain, $reply_target);

	if ($direction eq "out") {
		$start_chain = "FROM_" . uc($SUBNET);
		$start_target = "XBAR";
		$reply_chain = "TO_" . uc($SUBNET);
		$reply_target = "ACCEPT";

	}
	elsif ($direction eq "in") {
		$start_chain = "TO_" . uc($SUBNET);
		$start_target = "ACCEPT";
		$reply_chain = "FROM_" . uc($SUBNET);
		$reply_target = "XBAR";
	}
	elsif ($direction eq "local_in") {
		$start_chain = "INPUT";
		$start_target = "ACCEPT";
		$reply_chain = "OUTPUT";
		$reply_target = "ACCEPT";
	}
	elsif ($direction eq "local_out") {
		$start_chain = "OUTPUT";
		$start_target = "ACCEPT";
		$reply_chain = "INPUT";
		$reply_target = "ACCEPT";
	}
	else {
		print STDERR
			"ERROR: $SUBNET:$LINENUM: Direction must be in or out or local_in or local_out\n";
		return;
	}
	return  ($start_chain, $start_target, $reply_chain, $reply_target);
}


#  Outbound for generic protocols (i.e. non tcp/udp/icmp)
#sub allow_out_proto {
#	my ($proto, $dst) = @_;
#}
