#!/usr/bin/perl
#
# $Id: network_setup.pl 3 2006-02-07 12:52:43Z gregor $
#
#Copyright (c) 2006 Gregor Maier <gregor@majordomus.org>
#All rights reserved.
#
#
#Redistribution and use in source and binary forms, with or without
#modification, are permitted provided that the following conditions
#are met:
#
#1. Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#2. Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in the
#   documentation and/or other materials provided with the distribution.
#3. Neither the names of the copyright owners nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
#WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
#MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
#IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
#DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
#DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
#GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
#INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
#IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
#OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
#ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#
#
#
#
#
#
#
use strict;

return 1;

sub network_setup {
	my $cmd;
	my ($cursubnet, $curconfig, $curiface4, $curiface6);

	
	&network_setup_vlan;
	

	# Flush routing tables 
	gmsystem("$IPROUTE route flush root 0/0");
	gmsystem("$IPROUTE -6 route flush root ::/0");

	# Bring up lo loopback interface. Don't use ifup(), since we don't
	# want to flush the addresses on this iface ;-)
	gmsystem("$IPROUTE link set dev lo up");
	#
	# Bring up all other interfaces. 
	#
	foreach $cursubnet (keys %CONFIG) {
		($curiface4, $curiface6) = extract_iface($CONFIG{$cursubnet}, $cursubnet);
		if ($curiface4 ne "") {
			ifup($curiface4);
		}
		# If iface6 differs from iface4: bring it up
		if ($curiface6 ne "" and $curiface4 ne $curiface6) {
			ifup($curiface6);
		}
	}
	#
	# Configure addresses. 
	#
	foreach my $cursubnet (keys %CONFIG) {
		$curconfig = $CONFIG{$cursubnet};
		($curiface4, $curiface6) = extract_iface($curconfig, $cursubnet);
		my ($ip4,$ip6) = extract_ip($curconfig, $cursubnet);
		print "DEBUG: $curiface4 -- $curiface6: $ip4 ---- $ip6\n";
		set_ip($curiface4, $ip4);
		set_ip($curiface6, $ip6);

		#
		# If current subnet type is xfer, then add entries to the routing
		# table for the connected prefix.
		#
		if ($curconfig->{'type'} eq "xfer") {
			if ($curconfig->{'net4'} ne "" and $curconfig->{'nexthop4'} ne "") {
				$cmd = "$IPROUTE route replace to ". $curconfig->{'net4'} . " via " . $curconfig->{'nexthop4'};
				#print "$cmd\n";
				gmsystem($cmd);
			}
			if ($curconfig->{'net6'} ne "" and $curconfig->{'nexthop6'} ne "") {
				$cmd = "$IPROUTE -6 route replace to ". $curconfig->{'net6'} . " via " . $curconfig->{'nexthop6'};
				#print "$cmd\n";
				gmsystem($cmd);
			}
		}
	} # end foreach cursubnet

} # end setwork_setup


sub network_setup_vlan {
	my $cursubnet;
	my ($curiface4, $curiface6) ;
	my $cmd;

	# Bring vlan carrier interface up 
	ifup($VLAN_CARRIER_IFACE);

	# Set Naming style for vlan interfaces
	gmsystem("$VCONFIG set_name_type  VLAN_PLUS_VID_NO_PAD");


	#
	# Create VLAN inferfaces if required.
	#
	foreach $cursubnet (keys %CONFIG) {
		($curiface4, $curiface6) = extract_iface($CONFIG{$cursubnet}, $cursubnet);
		if ($curiface4 ne "") {
			# If it's a vlan interface: configure it. 
			if ($curiface4 =~ /^vlan(.*)/ and ! -e "/proc/net/vlan/$curiface4") {
				$cmd="$VCONFIG add $VLAN_CARRIER_IFACE $1";
				gmsystem($cmd);
			} 
		}
		# If iface6 differs from iface4: bring it up
		if ($curiface6 ne "" and $curiface4 ne $curiface6) {
			# If it's a vlan interface: configure it. 
			if ($curiface6 =~ /^vlan(.*)/ and ! -e "/proc/net/vlan/$curiface6") {
				$cmd="$VCONFIG add $VLAN_CARRIER_IFACE $1";
				gmsystem($cmd);
			} 
		}
	}
}


sub ifup {
	my $curiface = shift;
	my $cmd;
	
	print "$curiface\n";
	if ($curiface eq "") {
		print "ERROR: Trying to bring up unamed iface\n";
	}
	else {
		$cmd="$IPROUTE link set $curiface up" ;
		gmsystem($cmd);
		# Remove all IP-Addresses 
		# ip addr flush produces message on stderr if the iface doesn't have any
		# addresses but we are not interested in these messages
		# And the guys forgot the --silent option *grumpl*
		$cmd="$IPROUTE addr flush dev $curiface scope global 2>/dev/null";
		gmsystem($cmd);
	}
}


# Get's a CONFIG stanza as parameter
# Returns a list of ($iface4, $iface6), where iface4 and iface6 are 
# the interfaces for IPv4 and Ipv6 traffic for this subnet. 
# They will almost always be equal.!!!
sub extract_iface {
	my ($conf, $subnet) = @_;
	my ($iface4, $iface6);

	if ($conf->{'iface'} ne "") {
		if ( $conf->{'iface4'} ne "" || $conf->{'iface6'} ne "") {
			print STDERR "WARNING: Config for $subnet: iface is defined together with iface4 and/or iface6!\n";
		}
		$iface4 = $iface6= $conf->{'iface'};
	}
	else {
		$iface4 = $conf->{'iface4'};
		$iface6 = $conf->{'iface6'};
	}
	if ($iface4 eq "" and $iface6 eq "") {
			print STDERR "ERROR: Config for $subnet: iface is defined together with iface4 and/or iface6!\n";
	}
	return ($iface4, $iface6);
}

# Get's a CONFIG stanza as parameter
# Returns a list of ($ip4, $ip6), where ip4 and ip6 are IP-adresses
# with an appended prefixlen (i.e. with /25 or /64 etc. appended to the
# address. can be passed directly to the iproute tool
#
# If no address was configured, then undef is returned!
sub extract_ip {
	my ($conf, $cursubnet) = @_;
	my ($ip4, $ip6);
	my $prefixlen;

	if ($conf->{'type'} eq 'xfer') {
		# it's an transfer network. The IP-Addresses must be in the form IP/prefixlen
		# We assume that this is the case 
		return ($conf->{'ip4'}, $conf->{'ip6'});	
	}
	else {
		#
		# IPv4 
		#
		# Check if the IPv4 Address is specified with or without bitmask
		($ip4,$prefixlen) = split /\//, $conf->{'ip4'};
		if ($prefixlen eq "") {
			# specified without bitmask -- extrac the mask
			# the mask from the subnet. 
			(undef, $prefixlen) =  split /\//, $conf->{'net4'};
		}
		# if prefixlen is still empty that's an error (excpet, when no IPv4 address is configured)
		if ($prefixlen eq "" and $ip4 ne "") {
			print STDERR "ERROR: Could not find prefixlen/netmask for subnet $cursubnet\n";
			return (undef,undef);
		}
		$ip4 = "$ip4/$prefixlen";
		#
		# IPv6
		#
		($ip6,$prefixlen) = split /\//, $conf->{'ip6'};
		if ($prefixlen eq "") {
			# specified without bitmask -- extrac the mask
			# the mask from the subnet. 
			(undef, $prefixlen) =  split /\//, $conf->{'net6'};
		}
		# if prefixlen is still empty that's an error (excpet, when no IPv6 address is configured)
		if ($prefixlen eq "" and $ip6 ne "") {
			print STDERR "ERROR: Could not find prefixlen/netmask for subnet $cursubnet\n";
			return (undef,undef);
		}
		$ip6 = "$ip6/$prefixlen";

		return ($ip4, $ip6);
	}
}


sub set_ip {
	my $iface = shift;
	my $addr = shift;
	my $broadcast;

	if ($addr eq "" or $addr eq "/") {
		return;
	}
	# If it's an IPv4 address, we want to set a correct Broadcast address
	if ($addr =~ /^(\d{1,3}\.){3}\d{1,3}/) {
		$broadcast = "broadcast +";
	} 
	my $cmd="$IPROUTE addr add $addr $broadcast dev $iface";
	gmsystem($cmd);
}

