#!/usr/bin/env python

copyright = """
# Template python script: 
# Copyright (c) 2010 Gregor Maier <gregor@majordomus.org>
# Released under the modified BSD-license and GPL v2.0.
# see COPYING for details
#
# NOTE, NOTE, NOTE: The Mozilla Public Suffix list that might be embedded
# in this file as "publicSuffixRuleTree" is released under the Mozilla
# Public License. See below for copyright information.
#  
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
""";

import sys

"""
This is a module for finding the "effective TLD" (or "registered domain", or
"public suffix"). 

We utilize Mozilla's public suffix list: http://publicsuffix.org,

Some examples: 
   DNS name       "registered domain"   "effective TLD"
   foo.bar.com         bar                  com
   foo.bar.co.uk       bar                  co.uk


Internally we represent the public suffix rule list as a tree structure
consisting of Node objects. 

The modules interface is: 
   split_domainname(domain):
      return a 3-tuple of (sub_domain,reg_domain,effective_tld) where
      e.g., split_domainname("www.test.co.uk") would return:
          ("www", "test", "co.uk")

   get_registered_domain(domain):
      returns the registered domain including the publix suffix,
      e.g., get_registered_domain("www.test.co.uk") would return:
          "test.co.uk"
  
  get_registered_domain_part(domain):
      return registered domain without suffix. For the above example
      it would return "test"

  get_effective_tld(domain):
      return the TLD / public suffix. For the above example it would
      return "co.uk"
"""

class Node:
    """ 
    Class representing a node in the suffix rule tree. 
    
    * The chhildren dict() are the nodes children.  The children dict()'s keys
    are domain parts (e.g., uk, co, gv, com) and the values are Nodes()
    * The exclude flag specifies whether this (leaf) node is an exclusion rule
    (i.e., one with a leading "!") 
    * The hasstar flag specifies whether this nodes has a start '*' child (Note that
    the '*' is not added to the children dict())

    E.g., the rules 
       *.uk
       !nhs.uk
       at
       gv.at

    would map into the following tree:
       ROOT (hasstar=True, exclude=False)
         |- 'uk' (hasstar=True, exclude=False)
         |    `- 'nhs' (hasstar=False, exclude=True)
         |
         `- 'at' (hasstar=True, exclude=False)
              `- 'gv' (hasstar=False, exclude=False)
    """
    def __init__ (self, children=None, exclude=False, hasstar=False):
        self.children = children;
        self.exclude = exclude;
        self.hasstar = hasstar; 

    SHIFTWIDTH = 4
    def format (self, indent=1): 
        """ Return string (with newlines) that is a pretty representation of this 
        Node. Can be used for eval()"""
        nspaces = Node.SHIFTWIDTH * indent;
        spaces = " " * nspaces;
        if not self.children: 
            str = "Node(None, %s, %s)" % (repr(self.exclude), repr(self.hasstar) )
        else:
            str = "Node( {\n" + spaces;
            isfirstentry = True;
            for k, v in self.children.iteritems():
                if not isfirstentry:
                    str += ",\n" + spaces;
                isfirstentry = False;
                str += "'%s': %s" % (k, v.format(indent+1))
            str += "},\n" + spaces;
            str += "%s, %s)" % (repr(self.exclude), repr(self.hasstar))
        return str;

    def __repr__(self):
        return self.format();


def split_domainname(domain): 
    parts = domain.lower().split(".");

    curnode = publicSuffixRuleTree;

    idx = len(parts);
    for curpart in reversed(parts):
        if (not curnode.children) or (curpart not in curnode.children):
            # no further nodes to visit. 
            if curnode.exclude:
                # previous part is registered domain
                idx += 1;
            elif curnode.hasstar:
                # curpart is last part of effective TLD
                idx -= 1;
            else:
                # curpart is the registered domain name
                idx += 0;
            curnode = None;
            break;
        else:
            # more children
            curnode = curnode.children[curpart];
        idx -= 1;
    if curnode:
        # parts were exhausted before we reached a leaf
        if curnode.exclude:
            idx = 1;
        else:
            idx = 0;

    effective_tld = ".".join(parts[idx:]);
    reg_domain = "";
    sub_domain = "";
    if idx > 0:
        reg_domain = parts[idx-1]
    if idx > 1:
        sub_domain = ".".join(parts[:(idx-1)])

    return (sub_domain, reg_domain, effective_tld);


def get_registered_domain(domain):
    (sub_domain, reg_domain, effective_tld) = split_domainname(domain);
    if reg_domain == "":
        return "";
    else:
        return ".".join((reg_domain, effective_tld));

def get_registered_domain_part(domain):
    (sub_domain, reg_domain, effective_tld) = split_domainname(domain);
    return reg_domain;

def get_effective_tld(domain):
    (sub_domain, reg_domain, effective_tld) = split_domainname(domain);
    return effective_tld;
        


def main():
    for line in sys.stdin:
        line = line.strip();
        regdom = get_registered_domain(line);
        if regdom == "":
            print "error:", line;
        else:
            print regdom;
        


